package net.minecraft.block.state;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map.Entry;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;

import net.minecraft.block.Block;
import net.minecraft.block.properties.IProperty;

public abstract class BlockStateBase implements IBlockState {
	private static final Joiner COMMA_JOINER = Joiner.on(',');
	private static final Function<Entry<IProperty, Comparable>, String> MAP_ENTRY_TO_STRING = new Function<Entry<IProperty, Comparable>, String>() {
		public String apply(Entry<IProperty, Comparable> entry) {
			if (entry == null) {
				return "<NULL>";
			} else {
				IProperty iproperty = (IProperty) entry.getKey();
				return iproperty.getName() + "=" + iproperty.getName((Comparable) entry.getValue());
			}
		}
	};

	public <T extends Comparable<T>> IBlockState cycleProperty(IProperty<T> property) {
		return this.withProperty(property,
				(T) cyclePropertyValue(property.getAllowedValues(), this.getValue(property)));
	}

	protected static <T> T cyclePropertyValue(Collection<T> values, T currentValue) {
		Iterator iterator = values.iterator();

		while (iterator.hasNext()) {
			if (iterator.next().equals(currentValue)) {
				if (iterator.hasNext()) {
					return (T) iterator.next();
				}

				return (T) values.iterator().next();
			}
		}

		return (T) iterator.next();
	}

	public String toString() {
		StringBuilder stringbuilder = new StringBuilder();
		stringbuilder.append(Block.blockRegistry.getNameForObject(this.getBlock()));
		if (!this.getProperties().isEmpty()) {
			stringbuilder.append("[");
			COMMA_JOINER.appendTo(stringbuilder,
					Iterables.transform(this.getProperties().entrySet(), MAP_ENTRY_TO_STRING));
			stringbuilder.append("]");
		}

		return stringbuilder.toString();
	}
}