package net.minecraft.client.renderer.entity;

import java.util.Map;

import com.google.common.collect.Maps;

import net.lax1dude.eaglercraft.v1_8.opengl.GlStateManager;
import net.lax1dude.eaglercraft.v1_8.opengl.OpenGlHelper;
import net.lax1dude.eaglercraft.v1_8.opengl.WorldRenderer;
import net.lax1dude.eaglercraft.v1_8.opengl.ext.deferred.DeferredStateManager;
import net.minecraft.block.Block;
import net.minecraft.block.BlockBed;
import net.minecraft.block.state.IBlockState;
import net.minecraft.client.entity.AbstractClientPlayer;
import net.minecraft.client.gui.FontRenderer;
import net.minecraft.client.model.ModelChicken;
import net.minecraft.client.model.ModelCow;
import net.minecraft.client.model.ModelHorse;
import net.minecraft.client.model.ModelOcelot;
import net.minecraft.client.model.ModelPig;
import net.minecraft.client.model.ModelRabbit;
import net.minecraft.client.model.ModelSheep2;
import net.minecraft.client.model.ModelSlime;
import net.minecraft.client.model.ModelSquid;
import net.minecraft.client.model.ModelWolf;
import net.minecraft.client.model.ModelZombie;
import net.minecraft.client.renderer.RenderGlobal;
import net.minecraft.client.renderer.Tessellator;
import net.minecraft.client.renderer.culling.ICamera;
import net.minecraft.client.renderer.texture.TextureManager;
import net.minecraft.client.renderer.tileentity.RenderEnderCrystal;
import net.minecraft.client.renderer.tileentity.RenderItemFrame;
import net.minecraft.client.renderer.tileentity.RenderWitherSkull;
import net.minecraft.client.renderer.vertex.DefaultVertexFormats;
import net.minecraft.client.settings.GameSettings;
import net.minecraft.crash.CrashReport;
import net.minecraft.crash.CrashReportCategory;
import net.minecraft.entity.Entity;
import net.minecraft.entity.EntityLeashKnot;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.entity.ai.EntityMinecartMobSpawner;
import net.minecraft.entity.boss.EntityDragon;
import net.minecraft.entity.boss.EntityWither;
import net.minecraft.entity.effect.EntityLightningBolt;
import net.minecraft.entity.item.EntityArmorStand;
import net.minecraft.entity.item.EntityBoat;
import net.minecraft.entity.item.EntityEnderCrystal;
import net.minecraft.entity.item.EntityEnderEye;
import net.minecraft.entity.item.EntityEnderPearl;
import net.minecraft.entity.item.EntityExpBottle;
import net.minecraft.entity.item.EntityFallingBlock;
import net.minecraft.entity.item.EntityFireworkRocket;
import net.minecraft.entity.item.EntityItem;
import net.minecraft.entity.item.EntityItemFrame;
import net.minecraft.entity.item.EntityMinecart;
import net.minecraft.entity.item.EntityMinecartTNT;
import net.minecraft.entity.item.EntityPainting;
import net.minecraft.entity.item.EntityTNTPrimed;
import net.minecraft.entity.item.EntityXPOrb;
import net.minecraft.entity.monster.EntityBlaze;
import net.minecraft.entity.monster.EntityCaveSpider;
import net.minecraft.entity.monster.EntityCreeper;
import net.minecraft.entity.monster.EntityEnderman;
import net.minecraft.entity.monster.EntityEndermite;
import net.minecraft.entity.monster.EntityGhast;
import net.minecraft.entity.monster.EntityGiantZombie;
import net.minecraft.entity.monster.EntityGuardian;
import net.minecraft.entity.monster.EntityIronGolem;
import net.minecraft.entity.monster.EntityMagmaCube;
import net.minecraft.entity.monster.EntityPigZombie;
import net.minecraft.entity.monster.EntitySilverfish;
import net.minecraft.entity.monster.EntitySkeleton;
import net.minecraft.entity.monster.EntitySlime;
import net.minecraft.entity.monster.EntitySnowman;
import net.minecraft.entity.monster.EntitySpider;
import net.minecraft.entity.monster.EntityWitch;
import net.minecraft.entity.monster.EntityZombie;
import net.minecraft.entity.passive.EntityBat;
import net.minecraft.entity.passive.EntityChicken;
import net.minecraft.entity.passive.EntityCow;
import net.minecraft.entity.passive.EntityHorse;
import net.minecraft.entity.passive.EntityMooshroom;
import net.minecraft.entity.passive.EntityOcelot;
import net.minecraft.entity.passive.EntityPig;
import net.minecraft.entity.passive.EntityRabbit;
import net.minecraft.entity.passive.EntitySheep;
import net.minecraft.entity.passive.EntitySquid;
import net.minecraft.entity.passive.EntityVillager;
import net.minecraft.entity.passive.EntityWolf;
import net.minecraft.entity.projectile.EntityArrow;
import net.minecraft.entity.projectile.EntityEgg;
import net.minecraft.entity.projectile.EntityFishHook;
import net.minecraft.entity.projectile.EntityLargeFireball;
import net.minecraft.entity.projectile.EntityPotion;
import net.minecraft.entity.projectile.EntitySmallFireball;
import net.minecraft.entity.projectile.EntitySnowball;
import net.minecraft.entity.projectile.EntityWitherSkull;
import net.minecraft.init.Blocks;
import net.minecraft.init.Items;
import net.minecraft.util.AxisAlignedBB;
import net.minecraft.util.BlockPos;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.ReportedException;
import net.minecraft.util.Vec3;
import net.minecraft.world.World;

/**+
 * This portion of EaglercraftX contains deobfuscated Minecraft 1.8 source code.
 * 
 * Minecraft 1.8.8 bytecode is (c) 2015 Mojang AB. "Do not distribute!"
 * Mod Coder Pack v9.18 deobfuscation configs are (c) Copyright by the MCP Team
 * 
 * EaglercraftX 1.8 patch files are (c) 2022-2023 LAX1DUDE. All Rights Reserved.
 * 
 * WITH THE EXCEPTION OF PATCH FILES, MINIFIED JAVASCRIPT, AND ALL FILES
 * NORMALLY FOUND IN AN UNMODIFIED MINECRAFT RESOURCE PACK, YOU ARE NOT ALLOWED
 * TO SHARE, DISTRIBUTE, OR REPURPOSE ANY FILE USED BY OR PRODUCED BY THE
 * SOFTWARE IN THIS REPOSITORY WITHOUT PRIOR PERMISSION FROM THE PROJECT AUTHOR.
 * 
 * NOT FOR COMMERCIAL OR MALICIOUS USE
 * 
 * (please read the 'LICENSE' file this repo's root directory for more info) 
 * 
 */
public class RenderManager {
	private Map<Class<? extends Entity>, Render<? extends Entity>> entityRenderMap = Maps.newHashMap();
	private Map<String, RenderPlayer> skinMap = Maps.newHashMap();
	private RenderPlayer playerRenderer;
	private FontRenderer textRenderer;
	public static double renderPosX;
	public static double renderPosY;
	public static double renderPosZ;
	public TextureManager renderEngine;
	public World worldObj;
	public Entity livingPlayer;
	public Entity pointedEntity;
	public float playerViewY;
	public float playerViewX;
	public GameSettings options;
	public double viewerPosX;
	public double viewerPosY;
	public double viewerPosZ;
	private boolean renderOutlines = false;
	private boolean renderShadow = true;
	/**+
	 * whether bounding box should be rendered or not
	 */
	private boolean debugBoundingBox = false;

	public RenderManager(TextureManager renderEngineIn, RenderItem itemRendererIn) {
		this.renderEngine = renderEngineIn;
		this.entityRenderMap.put(EntityCaveSpider.class, new RenderCaveSpider(this));
		this.entityRenderMap.put(EntitySpider.class, new RenderSpider(this));
		this.entityRenderMap.put(EntityPig.class, new RenderPig(this, new ModelPig(), 0.7F));
		this.entityRenderMap.put(EntitySheep.class, new RenderSheep(this, new ModelSheep2(), 0.7F));
		this.entityRenderMap.put(EntityCow.class, new RenderCow(this, new ModelCow(), 0.7F));
		this.entityRenderMap.put(EntityMooshroom.class, new RenderMooshroom(this, new ModelCow(), 0.7F));
		this.entityRenderMap.put(EntityWolf.class, new RenderWolf(this, new ModelWolf(), 0.5F));
		this.entityRenderMap.put(EntityChicken.class, new RenderChicken(this, new ModelChicken(), 0.3F));
		this.entityRenderMap.put(EntityOcelot.class, new RenderOcelot(this, new ModelOcelot(), 0.4F));
		this.entityRenderMap.put(EntityRabbit.class, new RenderRabbit(this, new ModelRabbit(), 0.3F));
		this.entityRenderMap.put(EntitySilverfish.class, new RenderSilverfish(this));
		this.entityRenderMap.put(EntityEndermite.class, new RenderEndermite(this));
		this.entityRenderMap.put(EntityCreeper.class, new RenderCreeper(this));
		this.entityRenderMap.put(EntityEnderman.class, new RenderEnderman(this));
		this.entityRenderMap.put(EntitySnowman.class, new RenderSnowMan(this));
		this.entityRenderMap.put(EntitySkeleton.class, new RenderSkeleton(this));
		this.entityRenderMap.put(EntityWitch.class, new RenderWitch(this));
		this.entityRenderMap.put(EntityBlaze.class, new RenderBlaze(this));
		this.entityRenderMap.put(EntityPigZombie.class, new RenderPigZombie(this));
		this.entityRenderMap.put(EntityZombie.class, new RenderZombie(this));
		this.entityRenderMap.put(EntitySlime.class, new RenderSlime(this, new ModelSlime(16), 0.25F));
		this.entityRenderMap.put(EntityMagmaCube.class, new RenderMagmaCube(this));
		this.entityRenderMap.put(EntityGiantZombie.class, new RenderGiantZombie(this, new ModelZombie(), 0.5F, 6.0F));
		this.entityRenderMap.put(EntityGhast.class, new RenderGhast(this));
		this.entityRenderMap.put(EntitySquid.class, new RenderSquid(this, new ModelSquid(), 0.7F));
		this.entityRenderMap.put(EntityVillager.class, new RenderVillager(this));
		this.entityRenderMap.put(EntityIronGolem.class, new RenderIronGolem(this));
		this.entityRenderMap.put(EntityBat.class, new RenderBat(this));
		this.entityRenderMap.put(EntityGuardian.class, new RenderGuardian(this));
		this.entityRenderMap.put(EntityDragon.class, new RenderDragon(this));
		this.entityRenderMap.put(EntityEnderCrystal.class, new RenderEnderCrystal(this));
		this.entityRenderMap.put(EntityWither.class, new RenderWither(this));
		this.entityRenderMap.put(Entity.class, new RenderEntity(this));
		this.entityRenderMap.put(EntityPainting.class, new RenderPainting(this));
		this.entityRenderMap.put(EntityItemFrame.class, new RenderItemFrame(this, itemRendererIn));
		this.entityRenderMap.put(EntityLeashKnot.class, new RenderLeashKnot(this));
		this.entityRenderMap.put(EntityArrow.class, new RenderArrow(this));
		this.entityRenderMap.put(EntitySnowball.class, new RenderSnowball(this, Items.snowball, itemRendererIn));
		this.entityRenderMap.put(EntityEnderPearl.class, new RenderSnowball(this, Items.ender_pearl, itemRendererIn));
		this.entityRenderMap.put(EntityEnderEye.class, new RenderSnowball(this, Items.ender_eye, itemRendererIn));
		this.entityRenderMap.put(EntityEgg.class, new RenderSnowball(this, Items.egg, itemRendererIn));
		this.entityRenderMap.put(EntityPotion.class, new RenderPotion(this, itemRendererIn));
		this.entityRenderMap.put(EntityExpBottle.class,
				new RenderSnowball(this, Items.experience_bottle, itemRendererIn));
		this.entityRenderMap.put(EntityFireworkRocket.class, new RenderSnowball(this, Items.fireworks, itemRendererIn));
		this.entityRenderMap.put(EntityLargeFireball.class, new RenderFireball(this, 2.0F));
		this.entityRenderMap.put(EntitySmallFireball.class, new RenderFireball(this, 0.5F));
		this.entityRenderMap.put(EntityWitherSkull.class, new RenderWitherSkull(this));
		this.entityRenderMap.put(EntityItem.class, new RenderEntityItem(this, itemRendererIn));
		this.entityRenderMap.put(EntityXPOrb.class, new RenderXPOrb(this));
		this.entityRenderMap.put(EntityTNTPrimed.class, new RenderTNTPrimed(this));
		this.entityRenderMap.put(EntityFallingBlock.class, new RenderFallingBlock(this));
		this.entityRenderMap.put(EntityArmorStand.class, new ArmorStandRenderer(this));
		this.entityRenderMap.put(EntityMinecartTNT.class, new RenderTntMinecart(this));
		this.entityRenderMap.put(EntityMinecartMobSpawner.class, new RenderMinecartMobSpawner(this));
		this.entityRenderMap.put(EntityMinecart.class, new RenderMinecart(this));
		this.entityRenderMap.put(EntityBoat.class, new RenderBoat(this));
		this.entityRenderMap.put(EntityFishHook.class, new RenderFish(this));
		this.entityRenderMap.put(EntityHorse.class, new RenderHorse(this, new ModelHorse(), 0.75F));
		this.entityRenderMap.put(EntityLightningBolt.class, new RenderLightningBolt(this));
		this.playerRenderer = new RenderPlayer(this);
		this.skinMap.put("default", this.playerRenderer);
		this.skinMap.put("slim", new RenderPlayer(this, true, false));
		this.skinMap.put("zombie", new RenderPlayer(this, false, true));
	}

	public void setRenderPosition(double renderPosXIn, double renderPosYIn, double renderPosZIn) {
		this.renderPosX = renderPosXIn;
		this.renderPosY = renderPosYIn;
		this.renderPosZ = renderPosZIn;
	}

	public <T extends Entity> Render<T> getEntityClassRenderObject(Class<? extends Entity> parClass1) {
		Render render = (Render) this.entityRenderMap.get(parClass1);
		if (render == null && parClass1 != Entity.class) {
			render = this.getEntityClassRenderObject((Class<? extends Entity>) parClass1.getSuperclass());
			this.entityRenderMap.put(parClass1, render);
		}

		return render;
	}

	public <T extends Entity> Render getEntityRenderObject(Entity entityIn) {
		if (entityIn instanceof AbstractClientPlayer) {
			String s = ((AbstractClientPlayer) entityIn).getSkinType();
			RenderPlayer renderplayer = (RenderPlayer) this.skinMap.get(s);
			return renderplayer != null ? renderplayer : this.playerRenderer;
		} else {
			return this.<T>getEntityClassRenderObject(entityIn.getClass());
		}
	}

	public void cacheActiveRenderInfo(World worldIn, FontRenderer textRendererIn, Entity livingPlayerIn,
			Entity pointedEntityIn, GameSettings optionsIn, float partialTicks) {
		this.worldObj = worldIn;
		this.options = optionsIn;
		this.livingPlayer = livingPlayerIn;
		this.pointedEntity = pointedEntityIn;
		this.textRenderer = textRendererIn;
		if (livingPlayerIn instanceof EntityLivingBase && ((EntityLivingBase) livingPlayerIn).isPlayerSleeping()) {
			IBlockState iblockstate = worldIn.getBlockState(new BlockPos(livingPlayerIn));
			Block block = iblockstate.getBlock();
			if (block == Blocks.bed) {
				int i = ((EnumFacing) iblockstate.getValue(BlockBed.FACING)).getHorizontalIndex();
				this.playerViewY = (float) (i * 90 + 180);
				this.playerViewX = 0.0F;
			}
		} else {
			this.playerViewY = livingPlayerIn.prevRotationYaw
					+ (livingPlayerIn.rotationYaw - livingPlayerIn.prevRotationYaw) * partialTicks;
			this.playerViewX = livingPlayerIn.prevRotationPitch
					+ (livingPlayerIn.rotationPitch - livingPlayerIn.prevRotationPitch) * partialTicks;
		}

		if (optionsIn.thirdPersonView == 2) {
			this.playerViewY += 180.0F;
		}

		this.viewerPosX = livingPlayerIn.lastTickPosX
				+ (livingPlayerIn.posX - livingPlayerIn.lastTickPosX) * (double) partialTicks;
		this.viewerPosY = livingPlayerIn.lastTickPosY
				+ (livingPlayerIn.posY - livingPlayerIn.lastTickPosY) * (double) partialTicks;
		this.viewerPosZ = livingPlayerIn.lastTickPosZ
				+ (livingPlayerIn.posZ - livingPlayerIn.lastTickPosZ) * (double) partialTicks;
	}

	public void setPlayerViewY(float playerViewYIn) {
		this.playerViewY = playerViewYIn;
	}

	public boolean isRenderShadow() {
		return this.renderShadow;
	}

	public void setRenderShadow(boolean renderShadowIn) {
		this.renderShadow = renderShadowIn;
	}

	public void setDebugBoundingBox(boolean debugBoundingBoxIn) {
		this.debugBoundingBox = debugBoundingBoxIn;
	}

	public boolean isDebugBoundingBox() {
		return this.debugBoundingBox;
	}

	public boolean renderEntitySimple(Entity entityIn, float partialTicks) {
		return this.renderEntityStatic(entityIn, partialTicks, false);
	}

	public boolean shouldRender(Entity entityIn, ICamera camera, double camX, double camY, double camZ) {
		Render render = this.getEntityRenderObject(entityIn);
		return render != null && render.shouldRender(entityIn, camera, camX, camY, camZ);
	}

	public boolean renderEntityStatic(Entity entity, float partialTicks, boolean parFlag) {
		if (entity.ticksExisted == 0) {
			entity.lastTickPosX = entity.posX;
			entity.lastTickPosY = entity.posY;
			entity.lastTickPosZ = entity.posZ;
		}

		double d0 = entity.lastTickPosX + (entity.posX - entity.lastTickPosX) * (double) partialTicks;
		double d1 = entity.lastTickPosY + (entity.posY - entity.lastTickPosY) * (double) partialTicks;
		double d2 = entity.lastTickPosZ + (entity.posZ - entity.lastTickPosZ) * (double) partialTicks;
		float f = entity.prevRotationYaw + (entity.rotationYaw - entity.prevRotationYaw) * partialTicks;
		int i = entity.getBrightnessForRender(partialTicks);
		if (entity.isBurning()) {
			DeferredStateManager.setEmissionConstant(1.0f);
			i = 15728880;
		}

		int j = i % 65536;
		int k = i / 65536;
		OpenGlHelper.setLightmapTextureCoords(OpenGlHelper.lightmapTexUnit, (float) j / 1.0F, (float) k / 1.0F);
		GlStateManager.color(1.0F, 1.0F, 1.0F, 1.0F);
		try {
			return this.doRenderEntity(entity, d0 - this.renderPosX, d1 - this.renderPosY, d2 - this.renderPosZ, f,
					partialTicks, parFlag);
		} finally {
			DeferredStateManager.setEmissionConstant(0.0f);
		}
	}

	public static void setupLightmapCoords(Entity entity, float partialTicks) {
		int i = entity.getBrightnessForRender(partialTicks);
		if (entity.isBurning()) {
			DeferredStateManager.setEmissionConstant(1.0f);
			i = 15728880;
		}
		int j = i % 65536;
		int k = i / 65536;
		OpenGlHelper.setLightmapTextureCoords(OpenGlHelper.lightmapTexUnit, (float) j / 1.0F, (float) k / 1.0F);
	}

	public void renderWitherSkull(Entity entityIn, float partialTicks) {
		double d0 = entityIn.lastTickPosX + (entityIn.posX - entityIn.lastTickPosX) * (double) partialTicks;
		double d1 = entityIn.lastTickPosY + (entityIn.posY - entityIn.lastTickPosY) * (double) partialTicks;
		double d2 = entityIn.lastTickPosZ + (entityIn.posZ - entityIn.lastTickPosZ) * (double) partialTicks;
		Render render = this.getEntityRenderObject(entityIn);
		if (render != null && this.renderEngine != null) {
			int i = entityIn.getBrightnessForRender(partialTicks);
			int j = i % 65536;
			int k = i / 65536;
			OpenGlHelper.setLightmapTextureCoords(OpenGlHelper.lightmapTexUnit, (float) j / 1.0F, (float) k / 1.0F);
			GlStateManager.color(1.0F, 1.0F, 1.0F, 1.0F);
			render.renderName(entityIn, d0 - this.renderPosX, d1 - this.renderPosY, d2 - this.renderPosZ);
		}

	}

	public boolean renderEntityWithPosYaw(Entity entityIn, double x, double y, double z, float entityYaw,
			float partialTicks) {
		try {
			return this.doRenderEntity(entityIn, x, y, z, entityYaw, partialTicks, false);
		} finally {
			DeferredStateManager.setEmissionConstant(0.0f);
		}
	}

	public boolean doRenderEntity(Entity entity, double x, double y, double z, float entityYaw, float partialTicks,
			boolean parFlag) {
		Render render = null;

		try {
			render = this.getEntityRenderObject(entity);
			if (render != null && this.renderEngine != null) {
				try {
					if (render instanceof RendererLivingEntity) {
						((RendererLivingEntity) render).setRenderOutlines(this.renderOutlines);
					}

					RenderItem.renderPosX = (float) x;
					RenderItem.renderPosY = (float) y + entity.height * 0.5f;
					RenderItem.renderPosZ = (float) z;
					render.doRender(entity, x, y, z, entityYaw, partialTicks);
				} catch (Throwable throwable2) {
					throw new ReportedException(CrashReport.makeCrashReport(throwable2, "Rendering entity in world"));
				}

				try {
					if (!this.renderOutlines) {
						render.doRenderShadowAndFire(entity, x, y, z, entityYaw, partialTicks);
					}
				} catch (Throwable throwable1) {
					throw new ReportedException(
							CrashReport.makeCrashReport(throwable1, "Post-rendering entity in world"));
				}

				if (this.debugBoundingBox && !entity.isInvisible() && !parFlag
						&& !DeferredStateManager.isDeferredRenderer()) {
					try {
						this.renderDebugBoundingBox(entity, x, y, z, entityYaw, partialTicks);
					} catch (Throwable throwable) {
						throw new ReportedException(
								CrashReport.makeCrashReport(throwable, "Rendering entity hitbox in world"));
					}
				}
			} else if (this.renderEngine != null) {
				return false;
			}

			return true;
		} catch (Throwable throwable3) {
			CrashReport crashreport = CrashReport.makeCrashReport(throwable3, "Rendering entity in world");
			CrashReportCategory crashreportcategory = crashreport.makeCategory("Entity being rendered");
			entity.addEntityCrashInfo(crashreportcategory);
			CrashReportCategory crashreportcategory1 = crashreport.makeCategory("Renderer details");
			crashreportcategory1.addCrashSection("Assigned renderer", render);
			crashreportcategory1.addCrashSection("Location", CrashReportCategory.getCoordinateInfo(x, y, z));
			crashreportcategory1.addCrashSection("Rotation", Float.valueOf(entityYaw));
			crashreportcategory1.addCrashSection("Delta", Float.valueOf(partialTicks));
			throw new ReportedException(crashreport);
		}
	}

	/**+
	 * Renders the bounding box around an entity when F3+B is
	 * pressed
	 */
	private void renderDebugBoundingBox(Entity entityIn, double parDouble1, double parDouble2, double parDouble3,
			float parFloat1, float parFloat2) {
		GlStateManager.depthMask(false);
		GlStateManager.disableTexture2D();
		GlStateManager.disableLighting();
		GlStateManager.disableCull();
		GlStateManager.disableBlend();
		float f = entityIn.width / 2.0F;
		AxisAlignedBB axisalignedbb = entityIn.getEntityBoundingBox();
		AxisAlignedBB axisalignedbb1 = new AxisAlignedBB(axisalignedbb.minX - entityIn.posX + parDouble1,
				axisalignedbb.minY - entityIn.posY + parDouble2, axisalignedbb.minZ - entityIn.posZ + parDouble3,
				axisalignedbb.maxX - entityIn.posX + parDouble1, axisalignedbb.maxY - entityIn.posY + parDouble2,
				axisalignedbb.maxZ - entityIn.posZ + parDouble3);
		RenderGlobal.func_181563_a(axisalignedbb1, 255, 255, 255, 255);
		if (entityIn instanceof EntityLivingBase) {
			float f1 = 0.01F;
			RenderGlobal.func_181563_a(new AxisAlignedBB(parDouble1 - (double) f,
					parDouble2 + (double) entityIn.getEyeHeight() - 0.009999999776482582D, parDouble3 - (double) f,
					parDouble1 + (double) f, parDouble2 + (double) entityIn.getEyeHeight() + 0.009999999776482582D,
					parDouble3 + (double) f), 255, 0, 0, 255);
		}

		Tessellator tessellator = Tessellator.getInstance();
		WorldRenderer worldrenderer = tessellator.getWorldRenderer();
		Vec3 vec3 = entityIn.getLook(parFloat2);
		worldrenderer.begin(3, DefaultVertexFormats.POSITION_COLOR);
		worldrenderer.pos(parDouble1, parDouble2 + (double) entityIn.getEyeHeight(), parDouble3).color(0, 0, 255, 255)
				.endVertex();
		worldrenderer.pos(parDouble1 + vec3.xCoord * 2.0D,
				parDouble2 + (double) entityIn.getEyeHeight() + vec3.yCoord * 2.0D, parDouble3 + vec3.zCoord * 2.0D)
				.color(0, 0, 255, 255).endVertex();
		tessellator.draw();
		GlStateManager.enableTexture2D();
		GlStateManager.enableLighting();
		GlStateManager.enableCull();
		GlStateManager.disableBlend();
		GlStateManager.depthMask(true);
	}

	/**+
	 * World sets this RenderManager's worldObj to the world
	 * provided
	 */
	public void set(World worldIn) {
		this.worldObj = worldIn;
	}

	public double getDistanceToCamera(double parDouble1, double parDouble2, double parDouble3) {
		double d0 = parDouble1 - this.viewerPosX;
		double d1 = parDouble2 - this.viewerPosY;
		double d2 = parDouble3 - this.viewerPosZ;
		return d0 * d0 + d1 * d1 + d2 * d2;
	}

	/**+
	 * Returns the font renderer
	 */
	public FontRenderer getFontRenderer() {
		return this.textRenderer;
	}

	public void setRenderOutlines(boolean renderOutlinesIn) {
		this.renderOutlines = renderOutlinesIn;
	}
}